var tmUtils = require('./timeUtils');
var holidaysGetter = {};
var Holidays = require('./holidays');

var includeString = "810a6deb-46f2-4fa8-b779-bb5c2a6b5577";
var excludeString = "bb74c170-b06f-4d2b-ab88-2f6bf9ce8068";
var maybeIncludeString = "e85aacb0-bd79-4091-a271-85de0cebf66d";

function wrapper(_epochSpan) {
  var epochSpan;
  var year;
  var startYear;
  var endYear;
  setEpocheSpan(_epochSpan);

  function setEpocheSpan(newEpochSpan) {
    epochSpan = newEpochSpan;
    startYear = epochSpan[0].getFullYear();
    endYear = epochSpan[1].getFullYear();
    for (var yr = startYear; yr <= endYear; yr++) {
      if (!holidaysGetter[yr])
        holidaysGetter[yr] = new Holidays(yr);
    }
    year = new Date(epochSpan[0].getTime()/2 + epochSpan[1].getTime()/2).getFullYear();
  }

  function parseJSON(json) {
    var res = {};
    if (Array.isArray(json)) {
      res.incl = json[0];
      res.excl = json[1];
      res.maybe = json[2];
    }
    else if (json.incl){
      res = json;
    }
    else {
      res.incl = json[includeString];
      res.excl = json[excludeString];
      res.maybe = json[maybeIncludeString];
    }
    return res;
  }

  function getData(json, plz, cb) {
    json = parseJSON(json);

    getAllHolidays(plz, function (err, holidays) {
      if (err)
        return cb(err);

      var holidayIntervals = holidays.map(function (date) {
        return tmUtils.singleDay(date);
      });
      holidayIntervals = tmUtils.andOp(holidayIntervals, [epochSpan]);
      tmUtils.sort(holidayIntervals);

      try {
        var ret = {holidays: holidays};
        ret.intervals = buildOpeningHours([json.incl, json.excl], holidayIntervals);
        ret.maybeIntervals = [];
        if(json.maybe) {
          var maybeIntervals = buildOpeningHours([json.maybe], holidayIntervals);
          ret.maybeIntervals = tmUtils.andNotOp(maybeIntervals, ret.intervals);
         }
        cb(null, ret);
      }

      catch (err) {
        cb(err);
      }

    })
  }

  function buildOpeningHours(json, allHolidayIntervals) {
    var jsonIncl;
    var jsonExcl;

    jsonIncl = json[0];
    jsonExcl = json[1];

    if (!jsonIncl) {
      throw new Error('no openinghours model found');
    }

    var resultIncl = [];
    var resultExcl = [];
    var holidayIncl = [];

    var data, option, key;
    try {
      for (key in jsonIncl) {
        data = jsonIncl[key];
        option = getOption(data);
        if (option == "holiday")
          holidayIncl = holidayIncl.concat(getHolidays(data));
        else if (option == "vacation") {
          resultExcl.push(getOpeningHours(data, option));
        }
        else
          resultIncl.push(getOpeningHours(data, option));
      }

      if (jsonExcl) {
        for (key in jsonExcl) {
          data = jsonExcl[key];
          option = getOption(data);
          resultExcl.push(getOpeningHours(data, option));
        }
        resultExcl = tmUtils.orOp(resultExcl);
      }
    }

    catch (err) {
      throw err;
    }

    holidayIncl = tmUtils.andOp(holidayIncl, [epochSpan]);

    resultIncl = tmUtils.orOp(resultIncl);
    resultIncl = tmUtils.andNotOp(resultIncl, allHolidayIntervals);
    resultIncl = tmUtils.orOp([resultIncl, holidayIncl]);

    var result = resultIncl;
    if (jsonExcl)
      result = tmUtils.andNotOp(resultIncl, resultExcl);

    return result;
  }

  function getDaySpan(data) {
    var dayoptionS = data["dayoption"]["0"];
    var dayoption = parseInt(dayoptionS);
    if (dayoption < 7) return [dayoption, dayoption];
    switch (dayoption) {
      case 7:
        return [0, 0];
      case 8:
        return [1, 7];
      case 9:
        return [1, 5];
      case 10:
        return [1, 6];
      default:
        throw new Error("Invalid day option: " + dayoptionS);
    }
  }

  function parseTime(time) {
    var split = time.split(":");
    return {
      h: parseInt(split[0]) || 0,
      m: parseInt(split[1]) || 0
    }
  }

  function parseDate(date, year, addOneDay) {
    if (date.length == 0) {
      return null;
    }

    var _year;
    var split = date.split(".");
    var month = parseInt(split[1]) - 1;
    var day = parseInt(split[0]) + (addOneDay ? 1 : 0);
    if (split.length === 3) {
      _year = parseInt(split[2]);
    }
    if (!_year)
      _year = year;

    return new Date(_year, month, day);
  }

  function getHolidays(data) {
    var option = data["option"]["0"];
    if (option != "holiday") {
      throw new Error('not holiday: ' + option);
    }

    var timeSpan = getTimeSpan(data);
    var holidayId = data["holidayoption"][0];

    var days = [];
    for (var year = startYear; year <= endYear; year++) {
      days = days.concat(holidaysGetter[year].getHoliday(holidayId));
    }

    var intervals = days.map(function (date) {
      return tmUtils.singleDay(date, timeSpan);
    });
    tmUtils.sort(intervals);
    return intervals;
  }

  function getAllHolidays(plz, cb) {
    var days = [];
    var cnt = endYear - startYear;

    var errRec = false;

    function _cb(err, result) {
      if (errRec) return;
      if (err) {
        errRec = true;
        return cb(err);
      }

      days = days.concat(result);

      if (cnt-- === 0) {

        cb(null, days);
      }
    }

    for (var year = startYear; year <= endYear; year++) {
      holidaysGetter[year].getAllHolidays(plz, epochSpan, _cb);
    }
  }

  function getDateSpan(data) {

    var startDate = parseDate(data["opening_day_from"], year);
    var endDate = parseDate(data["opening_day_to"], year, true);

    if (!startDate)
      startDate = epochSpan[0];
    if (!endDate)
      endDate = epochSpan[1];

    //TODO: jahreswechsel beruecksichtigen
    return [startDate, endDate];
  }

  function getTimeSpan(data) {
    return [parseTime(data["opening_from"]), parseTime(data["opening_to"])];
  }

  function getOption(data) {
    return data["option"][0];
  }

  function getOpeningHours(data, option) {
    option = option || getOption(data);

    switch (option) {
      case "default":
        return getOpeningHoursDefault(data);
      case "season":
        return getOpeningHoursSeason(data);
      case "repeatable":
        return getOpeningHoursRepeatable(data);
      case "vacation":
        return getOpeningHoursVacation(data);
      case "":
        return [];
      default:
        throw new Error("option not supported: " + option);
        return [];
    }
  }

  function getOpeningHoursDefault(data) {
    var daySpan = getDaySpan(data);
    var timeSpan = getTimeSpan(data);
    return tmUtils.standardFromTillDay(daySpan, timeSpan, epochSpan, epochSpan, 7);
  }

  function getOpeningHoursSeason(data) {
    var daySpan = getDaySpan(data);
    var timeSpan = getTimeSpan(data);
    var dateSpan = getDateSpan(data);
    return tmUtils.standardFromTillDay(daySpan, timeSpan, dateSpan, epochSpan, 7);
  }

  function getOpeningHoursVacation(data) {
    if(data["opening_day_from"] == "" || data["opening_day_to"] == "") {
      console.error(new Error("opening_day cant be empty in vacations"));
    }
    var dateSpan = getDateSpan(data);
    return tmUtils.dateSpan(dateSpan);
  }

  function getOpeningHoursRepeatable(data) {
    var daySpan = getDaySpan(data);
    var timeSpan = getTimeSpan(data);
    var dateSpan = getDateSpan(data);
    var repeatOption = parseInt(data["repeatoption"]["0"]);

    var noInMonth;
    if (repeatOption <= 4) noInMonth = repeatOption;
    else if (repeatOption == 30) noInMonth = -1;
    if (noInMonth)
      return tmUtils.cyclicFromTillDay(daySpan, noInMonth, timeSpan, dateSpan, epochSpan);
    else {
      switch (repeatOption) {
        case 14:
        case 21:
        case 28:
          return tmUtils.standardFromTillDay(daySpan, timeSpan, dateSpan, epochSpan, repeatOption);
        default:
          throw new Error("invalid repeatoption");
      }
    }
  }

  return {
    getData: getData,
    setEpocheSpan: setEpocheSpan
  };
}
module.exports = wrapper;



