const holidayMapping = require('./holidayMapping.json');
const easterSunday = require('./easterSunday');
const DateHolidays = require('date-holidays');

let requestHandler = require('./regionInfoRequest');

const easterSundays = {};
const fixedHolidays = {
  101: '0101',
  106: '0601',
  501: '0105',
  808: '0808',
  815: '1508',
  103: '0310',
  131: '3110',
  111: '0111',
  121: '2412',
  122: '2512',
  123: '2612'
};

const variableOffsets = {
  301: -2,
  302: 0,
  303: 1,
  502: 39,
  601: 49,
  602: 50,
  603: 60
};

const holidayCache = {};
function getDateHolidays(countryCode) {
  if (!(countryCode in holidayCache)) {
    holidayCache[countryCode] = new DateHolidays(countryCode).getHolidays(2024);
  }

  return holidayCache[countryCode];
}

function Holidays(year) {
  function getEasterSunday() {
    if (!easterSundays[year]) {
      easterSundays[year] = easterSunday.getEasterSunday(year);
    }
    return easterSundays[year];
  }

  function getFixedHolidays(fixedHolidays, object) {
    const result = object || {};
    for (const key in fixedHolidays) {
      const dateString = fixedHolidays[key];
      const date = parseInt(dateString.slice(0, 2));
      const month = parseInt(dateString.slice(2, 4));
      result[key] = new Date(year, month - 1, date);
    }
    return result;
  }

  function getVariableHolidaysMap(offsets, object) {
    const result = object || {};
    const easterDate = getEasterSunday();
    for (const key in offsets) {
      const offset = offsets[key];
      result[key] = new Date(year, easterDate.getMonth(), easterDate.getDate() + offset);
    }
    return result;
  }

  let holidays = getFixedHolidays(fixedHolidays);
  holidays = getVariableHolidaysMap(variableOffsets, holidays);

  function getIsAugsburg(plz) {
    return !!(86150 <= plz && plz <= 86199);
  }

  function isInEpochSpan(date, epochSpan) {
    const dateTime = date.getTime();
    return epochSpan[0].getTime() <= dateTime && dateTime <= epochSpan[1].getTime();
  }

  function getRelevantHolidayKeys(epochSpan) {
    return Object.keys(holidays).filter(function (key) {
      return isInEpochSpan(holidays[key], epochSpan);
    });
  }

  function getHolidayKeys(info, epochSpan, cb) {
    const initKeys = getRelevantHolidayKeys(epochSpan);

    const specialKeys = holidayMapping['specificHolidays'].filter(function (key) {
      return initKeys.indexOf(key) >= 0;
    });

    if (specialKeys.length == 0) {
      return cb(null, initKeys);
    }

    requestHandler.getBundesland(info, function (err, blnd) {
      if (err) {
        return cb(err);
      }
      const keys = holidayMapping['sureHolidays'].filter(function (key) {
        return initKeys.indexOf(key) >= 0;
      });
      const regionMap = holidayMapping['region'][blnd];
      if (regionMap) {
        const regionHolidays = regionMap['default'];
        for (let i = 0; i < regionHolidays.length; i++) {
          const key = regionHolidays[i];
          if (initKeys.indexOf(key) >= 0) {
            keys.push(key);
          }
        }
      } else {
        console.error(new Error('No Mapping for Bundesland: ' + blnd));
      }
      if (blnd != 'BY') {
        cb(null, keys);
      } else {
        if (regionMap['augsburg'] && getIsAugsburg(info.zip)) {
          const augsHDs = regionMap['augsburg'];
          for (let i = 0; i < augsHDs.length; i++) {
            const key = augsHDs[i];
            if (initKeys.indexOf(key) >= 0) {
              keys.push(key);
            }
          }
        }
        if (regionMap['kath'] && regionMap['kath'].length > 0) {
          requestHandler.getKatholisch(info, function (err, kath) {
            if (err) {
              return cb(err);
            }
            if (kath) {
              const kathHDs = regionMap['kath'];
              for (let i = 0; i < kathHDs.length; i++) {
                const key = kathHDs[i];
                if (initKeys.indexOf(key) >= 0) {
                  keys.push(key);
                }
              }
            }
            cb(null, keys);
          });
        } else {
          cb(null, keys);
        }
      }
    });
  }

  this.getAllHolidays = function (info, epochSpan, cb) {
    if (info.countryCode && ['DE', 'AT', 'GB', 'US', 'CH', 'IT', 'LU', 'TZ'].includes(info.countryCode)) {
      const holidayDays = getDateHolidays(info.countryCode)
        .filter(x => x.type === 'public')
        .map(x => {
          const dateDay = x.date.slice(0, 10); // strip hours

          const date = new Date(`${dateDay}T00:00`); // converts 2021-05-13 to "2021-05-12T22:00:00.000Z"

          return date;
        });

      return cb(null, holidayDays);
    }

    getHolidayKeys(info, epochSpan, function (err, keys) {
      if (err) {
        return cb(err);
      }
      const result = [];
      for (let i = 0; i < keys.length; i++) {
        const key = keys[i];
        const elem = holidays[key];
        if (elem) {
          result.push(elem);
        }
      }
      cb(err, result);
    });
  };

  this.getHoliday = function (id) {
    if (id == 0) {
      const ret = [];
      for (const key in holidays) {
        ret.push(holidays[key]);
      }
      return ret;
    }
    if (id in holidays) {
      return [holidays[id]];
    }
    console.error('invalid holiday id ' + id);
    return null;
  };
}

module.exports = Holidays;

/**
 * set the object for handling the requests figuring out the bundesland and katholic
 * for testing reasons
 * @param handler
 */
module.exports.setRequestHandler = function (handler) {
  requestHandler = handler;
};

module.exports.setTestHandler = function () {
  requestHandler = {
    getBundesland: function (plz, cb) {
      //      console.log("bundesland is bayern");
      cb(null, 'BY');
    },
    getKatholisch: function (plz, cb) {
      //      console.log("is katholic");
      cb(null, true);
    }
  };
};
