"use strict";

var tmUtils = require('./timeUtils');
var holidaysGetter = {};
var Holidays = require('./holidays');

var includeString = "810a6deb-46f2-4fa8-b779-bb5c2a6b5577";
var excludeString = "bb74c170-b06f-4d2b-ab88-2f6bf9ce8068";
var maybeIncludeString = "e85aacb0-bd79-4091-a271-85de0cebf66d";

function wrapper(_epochSpan) {
  var debugInfo = '';

  var epochSpan;
  var year;
  var startYear;
  var endYear;
  setEpocheSpan(_epochSpan);

  function setEpocheSpan(newEpochSpan) {
    epochSpan = newEpochSpan;
    startYear = epochSpan[0].getFullYear();
    endYear = epochSpan[1].getFullYear();
    for (var yr = startYear; yr <= endYear; yr++) {
      if (!holidaysGetter[yr])
        holidaysGetter[yr] = new Holidays(yr);
    }
    year = new Date(epochSpan[0].getTime() / 2 + epochSpan[1].getTime() / 2).getFullYear();
  }

  function parseJSON(data) {
    var rawCustom = data["custom"];
    var custom = {};
    for (var key in rawCustom) {
      var field = rawCustom[key];
      custom[field] = {
        include: field.include || [],
        exclude: field.exclude || []
      }
    }

    return {
      incl: data["include"],
      excl: data["exclude"],
      maybe: data["maybe"],
      custom: custom
    }
  }

  function getData(info, cb) {
    var json = parseJSON(info.data);

    getAllHolidays(info, function (err, holidays) {
      if (err)
        return cb(err);

      var holidayIntervals = holidays.map(function (date) {
        return tmUtils.singleDay(date);
      });
      tmUtils.sort(holidayIntervals);
      holidayIntervals = tmUtils.andOp(holidayIntervals, [epochSpan]);

      try {
        var ret = {holidays: holidays};
        ret.intervals = buildOpeningHours([json.incl, json.excl], holidayIntervals);
        ret.maybeIntervals = [];
        if (json.maybe) {
          var maybeIntervals = buildOpeningHours([json.maybe], holidayIntervals);
          ret.maybeIntervals = tmUtils.andNotOp(maybeIntervals, ret.intervals);
        }
        ret.debugInfo = debugInfo;

        if (json.custom) {
          var custIntervals = ret.customIntervals = {};
          let custObj = json.custom;
          for (let key in custObj) {
            let field = custObj[key];
            custIntervals[key] = buildOpeningHours([field.include, field.exclude], []);
          }
        }

        cb(null, ret);
      }

      catch (err) {
        cb(err);
      }

    })
  }

  function buildOpeningHours(json, allHolidayIntervals) {
    var jsonIncl;
    var jsonExcl;

    jsonIncl = json[0];
    jsonExcl = json[1];

    if (!jsonIncl) {
      throw new Error('no openinghours model found');
    }

    var resultIncl = [];
    var resultExcl = [];
    var holidayIncl = [];

    var data, option, key;
    try {
      for (key in jsonIncl) {
        data = jsonIncl[key];
        option = getOption(data);
        if (option == "holiday")
          holidayIncl = holidayIncl.concat(getHolidays(data));
        else if (option == "vacation") {
          resultExcl.push(getOpeningHours(data, option));
        }
        else
          resultIncl.push(getOpeningHours(data, option));
      }

      if (jsonExcl) {
        for (key in jsonExcl) {
          data = jsonExcl[key];
          option = getOption(data);
          resultExcl.push(getOpeningHours(data, option));
        }
        resultExcl = tmUtils.orOp(resultExcl);
      }
    }

    catch (err) {
      throw err;
    }

    holidayIncl = tmUtils.andOp(holidayIncl, [epochSpan]);

    resultIncl = tmUtils.orOp(resultIncl);
    resultIncl = tmUtils.andNotOp(resultIncl, allHolidayIntervals);
    resultIncl = tmUtils.orOp([resultIncl, holidayIncl]);

    var result = resultIncl;
    if (jsonExcl)
      result = tmUtils.andNotOp(resultIncl, resultExcl);

    //result = tmUtils.splitMidnight(result);

    return result;
  }

  function getDaySpan(data) {
    var dayoptionS = data["dayoption"];
    var dayoption = parseInt(dayoptionS);
    if (dayoption < 7) return [dayoption, dayoption];
    switch (dayoption) {
      case 7:
        return [0, 0];
      case 8:
        return [1, 7];
      case 9:
        return [1, 5];
      case 10:
        return [1, 6];
      default:
        throw new Error("Invalid day option: " + dayoptionS);
    }
  }

  function parseTime(time) {
    time = time || "";
    var split = time.split(":");
    return {
      h: parseInt(split[0]) || 0,
      m: parseInt(split[1]) || 0
    }
  }

  function parseDate(date) {
    date = date || "";

    var split = date.split(".");
    if(!split || split.length < 2) {
      return null;
    }

    var month = parseInt(split[1]);
    var day = parseInt(split[0]);

    return [day, month];
  }

  function getHolidays(data) {
    var option = data["option"];
    if (option != "holiday") {
      throw new Error('not holiday: ' + option);
    }

    var timeSpan = getTimeSpan(data);
    var holidayId = data["holidayoption"];

    var days = [];
    for (var year = startYear; year <= endYear; year++) {
      days = days.concat(holidaysGetter[year].getHoliday(holidayId));
    }

    var intervals = days.map(function (date) {
      return tmUtils.singleDay(date, timeSpan);
    });
    tmUtils.sort(intervals);
    return intervals;
  }

  function getAllHolidays(info, cb) {
    var days = [];
    var cnt = endYear - startYear;

    var errRec = false;

    function _cb(err, result) {
      if (errRec) return;
      if (err) {
        errRec = true;
        return cb(err);
      }

      days = days.concat(result);

      if (cnt-- === 0) {

        cb(null, days);
      }
    }

    for (var year = startYear; year <= endYear; year++) {
      holidaysGetter[year].getAllHolidays(info, epochSpan, _cb);
    }
  }

  function getDateSpan(data) {
    var start = parseDate(data["opening_day_from"]);
    var end = parseDate(data["opening_day_to"]);

    if (!start) {
      return [epochSpan];
    }

    var startDate, endDate;

    if (!end) {
      debugInfo += 'no endDate for dateSpan';
      startDate = new Date(startYear, start[1] - 1, start[0]);
      endDate = new Date(startYear + 2, start[1] - 1, start[0]);
      return [[startDate, endDate]];
    }

    var newYear = 0;
    if (end[1] < start[1] || (end[1] == start[1] && end[0] < start[0])) newYear = 1;

    var dateSpanArray = [];
    for (var offset = -1; offset <= 1; offset++) {
      dateSpanArray.push([new Date(startYear + offset, start[1] - 1, start[0]),
        new Date(startYear + offset + newYear, end[1] - 1, end[0] + 1)])
    }
    return dateSpanArray;
  }

  function getTimeSpan(data) {
    return [parseTime(data["opening_from"]), parseTime(data["opening_to"])];
  }

  function getOption(data) {
    return data["option"];
  }

  function getOpeningHours(data, option) {
    option = option || getOption(data);

    switch (option) {
      case "default":
        return getOpeningHoursDefault(data);
      case "season":
        return getOpeningHoursSeason(data);
      case "repeatable":
        return getOpeningHoursRepeatable(data);
      case "vacation":
        return getOpeningHoursVacation(data);
      case "datespan":
        return getOpeningHoursDatespan(data);
      case "":
        return [];
      default:
        throw new Error("option not supported: " + option);
        return [];
    }
  }

  function getOpeningHoursDefault(data) {
    var daySpan = getDaySpan(data);
    var timeSpan = getTimeSpan(data);
    return tmUtils.standardFromTillDay(daySpan, timeSpan, epochSpan, epochSpan, 7);
  }

  function getOpeningHoursSeason(data) {
    var daySpan = getDaySpan(data);
    var timeSpan = getTimeSpan(data);
    var dateSpan = getDateSpan(data);
    return tmUtils.orOp(dateSpan.map(function (dateSpanElem) {
      return tmUtils.standardFromTillDay(daySpan, timeSpan, dateSpanElem, epochSpan, 7);
    }));
  }

  function getOpeningHoursDatespan(data) {
    // TODO JB remove entries beyond epochespan
    var from = new Date(data['opening_date_from']);
    var to   = new Date(data['opening_date_to']);
    if(!from || !to) {
      console.error(new Error('invalid data for datespan option ' + JSON.stringify(data)))
    }
    else {
      return [[from, to]];
    }
  }

  function getOpeningHoursVacation(data) {
    if (data["opening_day_from"] == "" || data["opening_day_to"] == "") {
      console.error(new Error("opening_day cant be empty in vacations"));
    }
    var dateSpan = getDateSpan(data);
    return tmUtils.orOp(dateSpan.map(function (dateSpanElem) {
      return tmUtils.dateSpan(dateSpanElem);
    }));
  }

  function getOpeningHoursRepeatable(data) {
    var daySpan = getDaySpan(data);
    var timeSpan = getTimeSpan(data);
    var dateSpan = getDateSpan(data);
    var repeatOption = parseInt(data["repeatoption"]);

    var noInMonth;
    if (repeatOption <= 4) noInMonth = repeatOption;
    else if (repeatOption == 30) noInMonth = -1;
    if (noInMonth)
      return tmUtils.orOp(dateSpan.map(function (dateSpanElem) {
        return tmUtils.cyclicFromTillDay(daySpan, noInMonth, timeSpan, dateSpanElem, epochSpan);
      }));
    else {
      switch (repeatOption) {
        case 14:
        case 21:
        case 28:
          return tmUtils.orOp(dateSpan.map(function (dateSpanElem) {
            return tmUtils.standardFromTillDay(daySpan, timeSpan, dateSpanElem, epochSpan, repeatOption);
          }));
        default:
          throw new Error("invalid repeatoption");
      }
    }
  }

  return {
    getData: getData,
    setEpocheSpan: setEpocheSpan
  };
}
module.exports = wrapper;



