var tmUtils = require('./timeUtils');
var Holidays = require('./holidays');
var now = new Date();
var holidaysGetter = {};


function wrapper(_epochSpan) {
  var epochSpan;
  var startYear;
  var endYear;
  setEpocheSpan(_epochSpan);

  function setEpocheSpan(newEpochSpan) {
    epochSpan = newEpochSpan;
    startYear = epochSpan[0].getFullYear();
    endYear = epochSpan[1].getFullYear();
    for(var year = startYear; year <= endYear; year++) {
      if(!holidaysGetter[year])
        holidaysGetter[year] = new Holidays(year);
    }
  }

  function calculateOpeningHours(json, plz, cb) {
    var includeString = "810a6deb-46f2-4fa8-b779-bb5c2a6b5577";
    var excludeString = "bb74c170-b06f-4d2b-ab88-2f6bf9ce8068";

    var jsonIncl = json[includeString];
    var jsonExcl = json[excludeString];

    if (!jsonIncl) {
      return cb(new Error('no openinghours model found'));
    }

    var resultIncl = [];
    var resultExcl = [];
    var holidayIncl = [];

    var data, option, key;
    try {
      for (key in jsonIncl) {
        data = jsonIncl[key];
        option = getOption(data);
        if (option == "holiday")
          holidayIncl = holidayIncl.concat(getHolidays(data));
//    else if (option == "vacation") {
//      resultExcl.push(getOpeningHours(data, option));
//    }
        else
          resultIncl.push(getOpeningHours(data, option));
      }

      if (jsonExcl) {
        for (key in jsonExcl) {
          data = jsonExcl[key];
          option = getOption(data);
          resultExcl.push(getOpeningHours(data, option));
        }
        resultExcl = tmUtils.orOp(resultExcl);
      }
    }
    catch (err) {
      return cb(err);
    }

    getAllHolidays(plz, function (err, allHolidays) {
      if (err)
        return cb(err);

      allHolidays = tmUtils.andOp(allHolidays, [epochSpan]);
      holidayIncl = tmUtils.andOp(holidayIncl, [epochSpan]);

      resultIncl = tmUtils.orOp(resultIncl);
      resultIncl = tmUtils.andNotOp(resultIncl, allHolidays);
      resultIncl = tmUtils.orOp([resultIncl, holidayIncl]);

      var result = resultIncl;
      if (jsonExcl)
        result = tmUtils.andNotOp(resultIncl, resultExcl);

      cb(null, result);
    });


  }

  function getDaySpan(data) {
    var dayoption = parseInt(data["dayoption"]["0"]);
    if (dayoption < 7) return [dayoption, dayoption];
    switch (dayoption) {
      case 7:
        return [0, 0];
      case 8:
        return [0, 7];
      case 9:
        return [1, 5];
      case 10:
        return [1, 6];
      default:
        throw new Error("Invalid day option: " + dayoption);
    }
  }

  function parseTime(time) {
    var split = time.split(":");
    return {
      h: parseInt(split[0]) || 0,
      m: parseInt(split[1]) || 0
    }
  }

  function parseDate(date, year, addOneDay) {
    if (date.length == 0)
      return null;

    var _year;
    var split = date.split(".");
    var month = parseInt(split[1]) - 1;
    var day = parseInt(split[0]) + (addOneDay ? 1 : 0);
    if (split.length === 3) {
      _year = parseInt(split[2]);
    }
    if (!_year)
      _year = year;

    return new Date(_year, month, day);
  }

  function getHolidays(data) {
    var option = data["option"]["0"];
    if (option != "holiday") {
      throw new Error('not holiday: ' + option);
    }

    var timeSpan = getTimeSpan(data);
    var holidayId = data["holidayoption"][0];

    var days = [];
    for (var year = startYear; year <= endYear; year++) {
      days = days.concat(holidaysGetter[year].getHoliday(holidayId));
    }

    var intervals = days.map(function (date) {
      return tmUtils.singleDay(date, timeSpan);
    });
    tmUtils.sort(intervals);
    return intervals;
  }

  function getAllHolidays(plz, cb) {
    var days = [];
    var cnt = endYear - startYear + 1;

    var errRec = false;
    function _cb(err, result) {
      if (errRec) return;
      if (err){
        errRec = true;
        return cb(err);
      }

      days = days.concat(result);
      cnt--;
      if (cnt === 0) {
        var intervals = days.map(function (date) {
          return tmUtils.singleDay(date);
        });
        tmUtils.sort(intervals);
        cb(null, intervals);
      }
    }

    for (var year = startYear; year <= endYear; year++) {
      holidaysGetter[year].getAllHolidays(plz, epochSpan, _cb);
    }
  }

  function getDateSpan(data) {
    var startDate = parseDate(data["opening_day_from"], now.getFullYear());
    var endDate = parseDate(data["opening_day_to"], now.getFullYear(), true);

    if (!startDate)
      startDate = epochSpan[0];
    if (!endDate)
      endDate = epochSpan[1];

    //TODO: jahreswechsel beruecksichtigen
    return [startDate, endDate];
  }

  function getTimeSpan(data) {
    return [parseTime(data["opening_from"]), parseTime(data["opening_to"])];
  }

  function getOption(data) {
    return data["option"][0];
  }

  function getOpeningHours(data, option) {
    option = option || getOption(data);

    switch (option) {
      case "default":
        return getOpeningHoursDefault(data);
      case "season":
        return getOpeningHoursSeason(data);
      case "repeatable":
        return getOpeningHoursRepeatable(data);
      case "vacation":
        return getOpeningHoursVacation(data);
      case "":
        return [];
      default:
        throw new Error("option not supported: " + option);
        return [];
    }
  }

  function getOpeningHoursDefault(data) {
    var daySpan = getDaySpan(data);
    var timeSpan = getTimeSpan(data);
    return tmUtils.standardFromTillDay(daySpan, timeSpan, epochSpan, epochSpan, 7);
  }

  function getOpeningHoursSeason(data) {
    var daySpan = getDaySpan(data);
    var timeSpan = getTimeSpan(data);
    var dateSpan = getDateSpan(data);
    return tmUtils.standardFromTillDay(daySpan, timeSpan, dateSpan, epochSpan, 7);
  }

  function getOpeningHoursVacation(data) {
    var dateSpan = getDateSpan(data);
    return tmUtils.dateSpan(dateSpan);
  }

  function getOpeningHoursRepeatable(data) {
    var daySpan = getDaySpan(data);
    var timeSpan = getTimeSpan(data);
    var dateSpan = getDateSpan(data);
    var repeatOption = parseInt(data["repeatoption"]["0"]);

    var noInMonth;
    if (repeatOption <= 4) noInMonth = repeatOption;
    else if (repeatOption == 30) noInMonth = -1;
    if (noInMonth)
      return tmUtils.cyclicFromTillDay(daySpan, noInMonth, timeSpan, dateSpan, epochSpan);
    else {
      switch (repeatOption) {
        case 14:
        case 21:
        case 28:
          return tmUtils.standardFromTillDay(daySpan, timeSpan, dateSpan, epochSpan, repeatOption);
        default:
          throw new Error("invalid repeatoption");
      }
    }
  }
  return {
    getData: calculateOpeningHours,
    setEpocheSpan: setEpocheSpan
  };
}
module.exports = wrapper;



